// ⚠️ IMPORTANT: This is the fixed server URL - DO NOT CHANGE!
const BASE = "https://web.officebaz.ir/google_map_data"; // root for our PHP endpoints

// Log server URL on startup
console.log('🌐 Extension configured for server:', BASE);
console.log('ℹ️ Server URL is locked and cannot be changed by users');

// Global state
const activeJobs = new Map();
let keepAliveInterval;
let isPaused = false;
let currentProgress = { total: 0, current: 0, skipped: 0 };
let notificationId = null;
let scrapingTabId = null;
let pollErrorCount = 0;
let lastErrorLogTime = 0;

// ---------- Service worker keep-alive ----------
chrome.runtime.onStartup.addListener(() => {
  console.log("🚀 Service worker started");
  startKeepAlive();
  initializeExtension();
});

chrome.runtime.onInstalled.addListener(() => {
  console.log("🔧 Extension installed/updated");
  startKeepAlive();
  initializeExtension();
});

function startKeepAlive() {
  if (keepAliveInterval) clearInterval(keepAliveInterval);
  keepAliveInterval = setInterval(() => {
    chrome.runtime.getPlatformInfo(() => {});
  }, 20000);
}

async function initializeExtension() {
  // Initialize storage
  chrome.storage.local.set({ 
    isPaused: false, 
    currentJob: null,
    stats: { total: 0, success: 0, failed: 0 }
  });
  
  // Create context menu for quick actions
  // Remove all existing menus first to avoid duplicates
  await chrome.contextMenus.removeAll();
  
  // Add a small delay to ensure cleanup is complete
  await new Promise(resolve => setTimeout(resolve, 100));
  
  try {
    chrome.contextMenus.create({
      id: "pause-scraping",
      title: "⏸️ Pause Scraping",
      contexts: ["all"]
    }, () => {
      if (chrome.runtime.lastError) {
        console.log("Menu already exists:", chrome.runtime.lastError.message);
      }
    });
    
    chrome.contextMenus.create({
      id: "resume-scraping",
      title: "▶️ Resume Scraping",
      contexts: ["all"]
    }, () => {
      if (chrome.runtime.lastError) {
        console.log("Menu already exists:", chrome.runtime.lastError.message);
      }
    });
  } catch (err) {
    console.log("Context menu creation error (can be ignored):", err);
  }
}

// Handle context menu clicks
chrome.contextMenus.onClicked.addListener((info) => {
  if (info.menuItemId === "pause-scraping") {
    isPaused = true;
    chrome.storage.local.set({ isPaused: true });
    showNotification("⏸️ Scraping Paused", "You can resume anytime from the extension icon.");
  } else if (info.menuItemId === "resume-scraping") {
    isPaused = false;
    chrome.storage.local.set({ isPaused: false });
    showNotification("▶️ Scraping Resumed", "Continuing to extract data...");
  }
});

// ---------- Helper: Show Chrome Notification ----------
function showNotification(title, message, priority = "normal") {
  chrome.notifications.create({
    type: "basic",
    iconUrl: "data:image/svg+xml,<svg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 100 100'><text y='75' font-size='75'>📍</text></svg>",
    title: title,
    message: message,
    priority: priority === "high" ? 2 : 0,
    requireInteraction: priority === "high"
  }, (id) => {
    notificationId = id;
  });
}

// ---------- Helper: Update Badge ----------
function updateBadge(text, color = "#4CAF50") {
  chrome.action.setBadgeText({ text: text.toString() });
  chrome.action.setBadgeBackgroundColor({ color: color });
}

// ---------- Helper functions ----------
function cleanPhoneNumber(phone) {
  if (!phone) return "";
  return phone.replace(/\s+/g, "").replace(/[^0-9+]/g, "");
}

// ---------- Helper function to extract coordinates from URL ----------
function extractCoordinatesFromUrl(url) {
  if (!url) return null;
  
  const match = url.match(/@(-?\d+\.?\d*),(-?\d+\.?\d*)/);
  if (match) {
    return {
      lat: parseFloat(match[1]),
      lng: parseFloat(match[2])
    };
  }
  return null;
}

// ---------- Helper function to check if coordinates are within bounds ----------
function isWithinBounds(coords, bounds) {
  if (!coords || !bounds) return true;
  
  const { lat, lng } = coords;
  const { north, south, east, west } = bounds;
  
  const withinLat = lat >= south && lat <= north;
  const withinLng = lng >= west && lng <= east;
  
  return withinLat && withinLng;
}

// ---------- Helper function to get city name from bounds ----------
function getCityNameFromBounds(bounds) {
  if (!bounds) return null;
  
  const { north, south, east, west } = bounds;
  
  const cities = [
    { name: 'شهرری', north: 35.6000, south: 35.5000, east: 51.5000, west: 51.3000 },
    { name: 'تهران', north: 35.8414, south: 35.6031, east: 51.6500, west: 51.1500 },
    { name: 'اصفهان', north: 32.7500, south: 32.5500, east: 51.8000, west: 51.6000 },
    { name: 'شیراز', north: 29.8000, south: 29.5000, east: 52.7000, west: 52.4000 },
    { name: 'تبریز', north: 38.2000, south: 38.0000, east: 46.4000, west: 46.1000 }
  ];
  
  for (const city of cities) {
    const tolerance = 0.01;
    if (Math.abs(north - city.north) < tolerance &&
        Math.abs(south - city.south) < tolerance &&
        Math.abs(east - city.east) < tolerance &&
        Math.abs(west - city.west) < tolerance) {
      return city.name;
    }
  }
  
  return null;
}

// ---------- Optimized Scraping Logic ----------
async function scrapeAll(tabId, searchArea = null) {
  console.log("🔎 Starting optimized scrape in tab:", tabId);
  if (searchArea) {
    console.log("📍 Using geographic bounds:", searchArea);
  }

  updateBadge("...", "#2196F3");

  // Optimized scroll with smart filtering
  const linksResult = await chrome.scripting.executeScript({
    target: { tabId },
    func: async (searchArea) => {
      const feed = document.querySelector("div[role='feed']");
      if (!feed) return [];
      
      let lastHeight = 0, sameCount = 0;
      let validResults = [];
      let maxScrolls = 50; // Increased for maximum coverage - can find up to 500+ results
      let scrollCount = 0;
      
      while (sameCount < 5 && scrollCount < maxScrolls) {
        // Smooth scroll instead of instant
        feed.scrollTo({ top: feed.scrollHeight, behavior: 'smooth' });
        await new Promise(r => setTimeout(r, 1200)); // Reduced wait time
        
        const currentCards = Array.from(feed.querySelectorAll("a.hfpxzc"));
        const currentResults = currentCards.map(a => a.getAttribute("href"));
        
        // Filter by geographic bounds if available
        if (searchArea) {
          const filteredResults = currentResults.filter(href => {
            if (!href) return false;
            
            const match = href.match(/@(-?\d+\.?\d*),(-?\d+\.?\d*)/);
            if (!match) return true;
            
            const lat = parseFloat(match[1]);
            const lng = parseFloat(match[2]);
            
            const withinLat = lat >= searchArea.south && lat <= searchArea.north;
            const withinLng = lng >= searchArea.west && lng <= searchArea.east;
            
            return withinLat && withinLng;
          });
          
          validResults = [...new Set(filteredResults)];
        } else {
          validResults = [...new Set(currentResults)];
        }
        
        if (feed.scrollHeight === lastHeight) {
          sameCount++;
        } else {
          sameCount = 0;
          lastHeight = feed.scrollHeight;
        }
        
        scrollCount++;
        
        // Early stop if we have enough results (increased limit)
        if (validResults.length >= 500 && searchArea) {
          console.log(`✅ Found ${validResults.length} valid results, stopping early`);
          break;
        }
        
        // Log progress
        if (scrollCount % 10 === 0) {
          console.log(`📊 Scroll ${scrollCount}/${maxScrolls}: Found ${validResults.length} results so far`);
        }
      }
      
      return validResults;
    },
    args: [searchArea]
  });

  const linkHrefs = linksResult[0].result;
  console.log(`✅ Found ${linkHrefs.length} businesses`);
  
  currentProgress.total = linkHrefs.length;
  currentProgress.current = 0;
  currentProgress.skipped = 0;

  const results = [];
  const seenPhones = new Set(); // Deduplicate by phone
  let batchResults = [];
  const BATCH_SIZE = 10; // Send data in batches
  
  // Get job_tag from activeJobs
  const jobInfo = activeJobs.get(tabId);
  const currentJobTag = jobInfo ? jobInfo.job_tag : 'unknown';

  for (let i = 0; i < linkHrefs.length; i++) {
    // Check if paused
    if (isPaused) {
      console.log("⏸️ Scraping paused at", i, "/", linkHrefs.length);
      showNotification("⏸️ Paused", `Progress saved: ${i}/${linkHrefs.length}`);
      await new Promise(resolve => {
        const checkPause = setInterval(() => {
          if (!isPaused) {
            clearInterval(checkPause);
            showNotification("▶️ Resumed", "Continuing extraction...");
            resolve();
          }
        }, 1000);
      });
    }

    currentProgress.current = i + 1;
    updateBadge(i + 1, "#4CAF50");
    console.log(`➡️ Processing ${i + 1}/${linkHrefs.length}`);

    // Click business card
    await chrome.scripting.executeScript({
      target: { tabId },
      func: (href) => {
        const card = Array.from(document.querySelectorAll("a.hfpxzc"))
                          .find(a => a.getAttribute("href") === href);
        if (card) card.click();
      },
      args: [linkHrefs[i]],
    });

    await new Promise(r => setTimeout(r, 2000)); // Reduced wait time

    // Extract details with better selectors
    const detailResult = await chrome.scripting.executeScript({
      target: { tabId },
      func: () => {
        function cleanPhoneNumber(phone) {
          if (!phone) return "";
          return phone.replace(/\s+/g, "").replace(/[^0-9+]/g, "");
        }
        
        const name = document.querySelector("h1.DUwDvf")?.innerText?.trim() || "";
        const address = document.querySelector('button[data-item-id^="address"] .Io6YTe')?.innerText?.trim() || "";
        const rawPhone = document.querySelector('button[data-item-id^="phone"] .Io6YTe')?.innerText?.trim() || "";
        const phone = cleanPhoneNumber(rawPhone);
        const rating = document.querySelector("div.F7nice span[aria-hidden='true']")?.innerText?.trim() || "";
        const website = document.querySelector('a[data-item-id^="authority"]')?.href || "";
        
        return { name, address, phone, rating, website };
      },
    });

    const data = detailResult[0].result;
    
    // Smart filtering: skip duplicates and invalid entries
    if (data && data.name && data.phone && !seenPhones.has(data.phone)) {
      seenPhones.add(data.phone);
      console.log("✅ Extracted:", data);
      results.push(data);
      batchResults.push(data);
      
      // Send batch to server
      if (batchResults.length >= BATCH_SIZE) {
        await sendBatchToServer(batchResults, currentJobTag);
        batchResults = [];
      }
    } else {
      currentProgress.skipped++;
      console.log("⏭️ Skipped: duplicate or invalid");
    }
  }

  // Send remaining batch
  if (batchResults.length > 0) {
    await sendBatchToServer(batchResults, currentJobTag);
  }

  console.log(`🎯 Final: ${results.length} extracted, ${currentProgress.skipped} skipped`);
  return results;
}

// ---------- Send batch to server ----------
async function sendBatchToServer(batch, job_tag) {
  try {
    const response = await fetch(`${BASE}/insert_places.php`, {
      method: "POST",
      headers: { 
        "Content-Type": "application/json",
        "Accept": "application/json"
      },
      credentials: "include",
      mode: "cors",
      body: JSON.stringify({ job_tag: job_tag, places: batch }),
    });
    
    const result = await response.json();
    console.log(`📤 Batch of ${batch.length} records saved`);
    return result;
  } catch (err) {
    console.error("❌ Batch save error:", err);
    throw err;
  }
}

// ---------- Poll server for jobs ----------
async function pollJobs() {
  if (isPaused) return; // Don't poll when paused
  
  try {
    const r = await fetch(`${BASE}/get_job.php`, { 
      credentials: "include",
      mode: "cors",
      headers: {
        'Content-Type': 'application/json'
      }
    });
    
    // Reset error count on successful connection
    pollErrorCount = 0;
    
    const j = await r.json();
    
    if (!j || !j.success || !j.job || !j.job.search_tag) return;

    const job_tag = j.job.search_tag;
    const searchArea = j.job.search_area;
    
    // Build search URL
    let url = `https://www.google.com/maps/search/${encodeURIComponent(job_tag)}`;
    
    if (searchArea && searchArea.north && searchArea.south && searchArea.east && searchArea.west) {
      const centerLat = (searchArea.north + searchArea.south) / 2;
      const centerLng = (searchArea.east + searchArea.west) / 2;
      
      const cityName = getCityNameFromBounds(searchArea);
      if (cityName) {
        url = `https://www.google.com/maps/search/${encodeURIComponent(job_tag + ' ' + cityName)}/@${centerLat},${centerLng},13z`;
      } else {
        url = `https://www.google.com/maps/search/${encodeURIComponent(job_tag)}/@${centerLat},${centerLng},13z`;
      }
      
      console.log('🗺️ Searching with bounds:', {
        center: `${centerLat},${centerLng}`,
        cityName: cityName,
        url: url
      });
    }

    // Show notification
    showNotification("🚀 Starting Extraction", `Searching for: ${job_tag}`);

    // Create tab in background (not active)
    chrome.tabs.create({ url, active: false }, async (tab) => {
      const tabId = tab.id;
      scrapingTabId = tabId;
      activeJobs.set(tabId, { job_tag, searchArea, windowId: tab.windowId });
      
      // Store current job
      chrome.storage.local.set({ currentJob: { job_tag, searchArea, tabId } });
      
      // Wait for page to load
      setTimeout(async () => {
        try {
          const results = await scrapeAll(tabId, searchArea);

          // Results are already sent via batch processing
          // No need to send again here
          console.log("✅ All records saved for tag:", job_tag, "Total:", results.length);
          
          // Show success notification
          showNotification(
            "✅ Extraction Complete!", 
            `Successfully extracted ${results.length} contacts for "${job_tag}"`,
            "high"
          );
          
          updateBadge("✓", "#4CAF50");
          
          // Close tab and cleanup
          chrome.tabs.remove(tabId);
          activeJobs.delete(tabId);
          chrome.storage.local.set({ currentJob: null });
          scrapingTabId = null;
          
          // Reset badge after 5 seconds
          setTimeout(() => updateBadge("", "#4CAF50"), 5000);
          
        } catch (err) {
          console.error("❌ Scraping error:", err);
          showNotification("❌ Error", "Failed to extract data. Please try again.", "high");
          updateBadge("✗", "#F44336");
          
          chrome.tabs.remove(tabId);
          activeJobs.delete(tabId);
          chrome.storage.local.set({ currentJob: null });
          scrapingTabId = null;
        }
      }, 6000);
    });
    
  } catch (e) {
    pollErrorCount++;
    
    // Only log error occasionally to avoid console spam
    const now = Date.now();
    const shouldLog = (now - lastErrorLogTime) > 30000; // Log every 30 seconds max
    
    if (shouldLog) {
      console.warn(`⚠️ Cannot connect to server (${pollErrorCount} attempts). Make sure ${BASE}/get_job.php is running.`);
      lastErrorLogTime = now;
      
      // Show notification only once after many failures
      if (pollErrorCount === 10) {
        showNotification(
          "⚠️ Server Offline",
          "Cannot connect to local server. Please start your PHP server at " + BASE,
          "normal"
        );
      }
    }
    
    // Update badge to show offline status after several failures
    if (pollErrorCount > 5 && activeJobs.size === 0) {
      updateBadge("⚠️", "#FF9800");
    }
  }
}

// ---------- Message listener for popup communication ----------
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === "getStatus") {
    // Get current job info
    let currentJobInfo = null;
    if (activeJobs.size > 0) {
      // Get first active job
      const firstJob = Array.from(activeJobs.values())[0];
      if (firstJob) {
        currentJobInfo = {
          tag: firstJob.job_tag,
          area: firstJob.searchArea
        };
      }
    }
    
    sendResponse({
      isPaused: isPaused,
      progress: currentProgress,
      hasActiveJob: activeJobs.size > 0,
      currentJob: currentJobInfo
    });
  } else if (request.action === "pause") {
    isPaused = true;
    chrome.storage.local.set({ isPaused: true });
    sendResponse({ success: true });
  } else if (request.action === "resume") {
    isPaused = false;
    chrome.storage.local.set({ isPaused: false });
    sendResponse({ success: true });
  } else if (request.action === "stop") {
    if (scrapingTabId) {
      chrome.tabs.remove(scrapingTabId);
      activeJobs.clear();
      scrapingTabId = null;
    }
    isPaused = false;
    currentProgress = { total: 0, current: 0, skipped: 0 };
    chrome.storage.local.set({ currentJob: null, isPaused: false });
    updateBadge("", "#4CAF50");
    sendResponse({ success: true });
  } else if (request.type === 'getServerUrl') {
    // Return fixed server URL (read-only)
    sendResponse({ url: BASE, locked: true });
  }
  
  return true; // Keep channel open for async response
});

// ---------- Start ----------
startKeepAlive();
initializeExtension();
setInterval(pollJobs, 5000);
