// Dynamic server detection - works for both local and production
let BASE = "https://web.officebaz.ir/google_map_data"; // Default to production

// Try to detect environment from storage
chrome.storage.local.get(['serverUrl'], function(result) {
  if (result.serverUrl) {
    BASE = result.serverUrl;
    console.log('✅ Using saved server URL:', BASE);
  } else {
    // Set default
    chrome.storage.local.set({ serverUrl: BASE });
    console.log('ℹ️ Using default server URL:', BASE);
  }
});

// Global state
const activeJobs = new Map();
let keepAliveInterval;
let isPaused = false;
let currentProgress = { total: 0, current: 0, skipped: 0 };
let notificationId = null;
let scrapingTabId = null;
let pollErrorCount = 0;
let lastErrorLogTime = 0;

// ---------- Service worker keep-alive ----------
chrome.runtime.onStartup.addListener(() => {
  console.log("🚀 Service worker started");
  startKeepAlive();
  initializeExtension();
});

chrome.runtime.onInstalled.addListener(() => {
  console.log("🔧 Extension installed/updated");
  startKeepAlive();
  initializeExtension();
});

function startKeepAlive() {
  if (keepAliveInterval) clearInterval(keepAliveInterval);
  keepAliveInterval = setInterval(() => {
    chrome.runtime.getPlatformInfo(() => {});
  }, 20000);
}

async function initializeExtension() {
  // Initialize storage
  chrome.storage.local.set({ 
    isPaused: false, 
    currentJob: null,
    stats: { total: 0, success: 0, failed: 0 }
  });
  
  // Create context menu for quick actions
  // Remove all existing menus first to avoid duplicates
  await chrome.contextMenus.removeAll();
  
  // Add a small delay to ensure cleanup is complete
  await new Promise(resolve => setTimeout(resolve, 100));
  
  try {
    chrome.contextMenus.create({
      id: "pause-scraping",
      title: "⏸️ Pause Scraping",
      contexts: ["all"]
    }, () => {
      if (chrome.runtime.lastError) {
        console.log("Menu already exists:", chrome.runtime.lastError.message);
      }
    });
    
    chrome.contextMenus.create({
      id: "resume-scraping",
      title: "▶️ Resume Scraping",
      contexts: ["all"]
    }, () => {
      if (chrome.runtime.lastError) {
        console.log("Menu already exists:", chrome.runtime.lastError.message);
      }
    });
  } catch (err) {
    console.log("Context menu creation error (can be ignored):", err);
  }
}

// Handle context menu clicks
chrome.contextMenus.onClicked.addListener((info) => {
  if (info.menuItemId === "pause-scraping") {
    isPaused = true;
    chrome.storage.local.set({ isPaused: true });
    showNotification("⏸️ Scraping Paused", "You can resume anytime from the extension icon.");
  } else if (info.menuItemId === "resume-scraping") {
    isPaused = false;
    chrome.storage.local.set({ isPaused: false });
    showNotification("▶️ Scraping Resumed", "Continuing to extract data...");
  }
});

// Helper function to get current BASE URL
async function getBaseUrl() {
  return new Promise((resolve) => {
    chrome.storage.local.get(['serverUrl'], function(result) {
      resolve(result.serverUrl || BASE);
    });
  });
}

// Helper: Show Chrome Notification
function showNotification(title, message, priority = "normal") {
  chrome.notifications.create({
    type: "basic",
    iconUrl: "data:image/svg+xml,<svg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 100 100'><text y='75' font-size='75'>📍</text></svg>",
    title: title,
    message: message,
    priority: priority === "high" ? 2 : 0,
    requireInteraction: priority === "high"
  }, (id) => {
    notificationId = id;
  });
}

// Update Badge
function updateBadge(text, color = "#4285F4") {
  chrome.action.setBadgeText({ text: String(text) });
  chrome.action.setBadgeBackgroundColor({ color: color });
}

// Message listener for settings updates
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.type === 'updateServerUrl') {
    BASE = message.url;
    chrome.storage.local.set({ serverUrl: message.url });
    console.log('✅ Server URL updated to:', BASE);
    showNotification('تنظیمات ذخیره شد', 'آدرس سرور با موفقیت تغییر کرد');
    sendResponse({ success: true });
  } else if (message.type === 'getServerUrl') {
    getBaseUrl().then(url => {
      sendResponse({ url: url });
    });
    return true; // Keep channel open for async response
  }
});

console.log('🎯 Extension loaded with dynamic server support');
console.log('📍 Default BASE:', BASE);

