// UI Elements
const statusBadge = document.getElementById('statusBadge');
const activeJob = document.getElementById('activeJob');
const progressContainer = document.getElementById('progressContainer');
const progressFill = document.getElementById('progressFill');
const progressPercent = document.getElementById('progressPercent');
const totalCount = document.getElementById('totalCount');
const currentCount = document.getElementById('currentCount');
const skippedCount = document.getElementById('skippedCount');

const pauseBtn = document.getElementById('pauseBtn');
const resumeBtn = document.getElementById('resumeBtn');
const stopBtn = document.getElementById('stopBtn');

// Panel link
const panelLink = document.getElementById('panelLink');

// Update UI from background
function updateUI() {
  chrome.runtime.sendMessage({ action: 'getStatus' }, (response) => {
    if (!response) return;
    
    const { isPaused, progress, hasActiveJob, currentJob } = response;
    
    // Update status badge
    if (hasActiveJob) {
      if (isPaused) {
        statusBadge.className = 'status-badge paused';
        statusBadge.innerHTML = '⏸️ متوقف شده';
      } else {
        statusBadge.className = 'status-badge active';
        statusBadge.innerHTML = '<span class="loading">●</span> در حال استخراج...';
      }
      activeJob.textContent = 'بله';
      
      // Show progress
      progressContainer.style.display = 'block';
      
      // Update progress bar
      const percent = progress.total > 0 ? Math.round((progress.current / progress.total) * 100) : 0;
      progressFill.style.width = percent + '%';
      progressPercent.textContent = percent + '%';
      
      // Update counts
      totalCount.textContent = progress.total;
      currentCount.textContent = progress.current;
      skippedCount.textContent = progress.skipped;
      
      // Enable/disable buttons
      pauseBtn.disabled = isPaused;
      resumeBtn.disabled = !isPaused;
      stopBtn.disabled = false;
      
    } else {
      statusBadge.className = 'status-badge idle';
      statusBadge.innerHTML = '<span class="loading">●</span> در انتظار...';
      activeJob.textContent = 'خیر';
      
      progressContainer.style.display = 'none';
      
      // Disable all buttons
      pauseBtn.disabled = true;
      resumeBtn.disabled = true;
      stopBtn.disabled = true;
    }
  });
}

// Event Listeners
pauseBtn.addEventListener('click', () => {
  chrome.runtime.sendMessage({ action: 'pause' }, () => {
    updateUI();
    showToast('⏸️ استخراج متوقف شد');
  });
});

resumeBtn.addEventListener('click', () => {
  chrome.runtime.sendMessage({ action: 'resume' }, () => {
    updateUI();
    showToast('▶️ استخراج ادامه یافت');
  });
});

stopBtn.addEventListener('click', () => {
  if (confirm('آیا مطمئن هستید که می‌خواهید استخراج را لغو کنید؟')) {
    chrome.runtime.sendMessage({ action: 'stop' }, () => {
      updateUI();
      showToast('⏹️ استخراج لغو شد');
    });
  }
});

// Toast notification (simple)
function showToast(message) {
  const toast = document.createElement('div');
  toast.textContent = message;
  toast.style.cssText = `
    position: fixed;
    bottom: 20px;
    left: 50%;
    transform: translateX(-50%);
    background: rgba(0, 0, 0, 0.8);
    color: white;
    padding: 10px 20px;
    border-radius: 8px;
    font-size: 13px;
    z-index: 9999;
    animation: fadeInOut 2s ease-in-out;
  `;
  document.body.appendChild(toast);
  
  setTimeout(() => {
    toast.remove();
  }, 2000);
}

// Add fadeInOut animation
const style = document.createElement('style');
style.textContent = `
  @keyframes fadeInOut {
    0% { opacity: 0; transform: translateX(-50%) translateY(10px); }
    20%, 80% { opacity: 1; transform: translateX(-50%) translateY(0); }
    100% { opacity: 0; transform: translateX(-50%) translateY(-10px); }
  }
`;
document.head.appendChild(style);

// Set panel link
chrome.runtime.sendMessage({ type: 'getServerUrl' }, (response) => {
  if (response && response.url) {
    panelLink.href = response.url;
    console.log('📍 Connected to server:', response.url);
  }
});

// Update UI every second
updateUI();
setInterval(updateUI, 1000);














