<?php
/**
 * Plugin Name: اطلاع رسانی آفیس باز
 * Plugin URI:  https://web.officebaz.ir/docs/?article=woo-plugin
 * Description: ارسال خودکار پیام متنی به واتساپ وقتی سفارش ووکامرس به وضعیت "تکمیل" رسید. شماره‌ها به‌طور خودکار به فرمت ایران (989123456789) نرمالایز می‌شوند.
 * Version:     2.0.0
 * Author:      آفیس باز
 * Text Domain: wc-whatsapp-sender
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

if ( ! class_exists( 'WC_WhatsApp_Sender' ) ) :

class WC_WhatsApp_Sender {

    const OPTION_KEY = 'wc_whatsapp_sender_options';
    const SENT_MESSAGES_KEY = 'wc_whatsapp_sent_messages';

    private $defaults = array(
        'api_url'      => 'https://web.officebaz.ir/api/send-text.php',
        'api_key'      => '',
        'sender'       => '',
        'phone_field'  => 'auto_search',
        'message'      => '👋 سلام {first_name} عزیز

🏪 {site_name}

📦 سفارش #{order_id}
💰 مجموع: {total}

🛍️ محصولات سفارش:
{products_list}

{status_message}

{last_notes}',
        'status_messages' => array(
            'pending'    => '⏳ منتظر پرداخت شما هستیم. لطفاً برای تکمیل سفارش، پرداخت را انجام دهید.',
            'processing' => '🔄 در حال انجام سفارش شما هستیم. ممنون از صبوری شما!',
            'completed'  => '✅ سفارش شما با موفقیت تکمیل شد و آماده ارسال است. ممنون از اعتماد شما!',
            'on-hold'    => '⏸️ سفارش شما در انتظار بررسی است. به زودی با شما تماس خواهیم گرفت.',
            'cancelled'  => '❌ متأسفانه سفارش شما لغو شد. در صورت نیاز با پشتیبانی تماس بگیرید.',
            'refunded'   => '💰 مبلغ سفارش شما بازپرداخت شد. در صورت نیاز با پشتیبانی تماس بگیرید.',
            'failed'     => '⚠️ متأسفانه مشکلی در پردازش سفارش شما رخ داد. لطفاً دوباره تلاش کنید.',
        ),
        'enabled'      => 'yes',
        'log_to_order' => 'yes',
        'add_country'  => '98', // پیشفرض: ایران
        'order_statuses' => array( 'pending', 'completed' ), // وضعیت‌های سفارش برای ارسال پیام
        'admin_notify' => 'yes', // اعلان مدیر
        'admin_phone'  => '', // شماره مدیر
        'admin_message' => '🔔 سفارش جدید #{order_id}

👤 مشتری: {customer_name}
📞 تلفن: {customer_phone}
📧 ایمیل: {customer_email}

📍 آدرس:
{shipping_address}

🛍️ محصولات:
{products_list}

💰 مجموع: {total}
📅 تاریخ: {order_date}
🔄 وضعیت: {status_text}

{status_message}',
        'sms_enabled'   => 'no',
        'sms_username'  => '',
        'sms_password'  => '',
        'sms_from'      => '',
        'sms_wsdl'      => 'http://api.payamak-panel.com/post/send.asmx?wsdl',
        'sms_message'   => '👋 {first_name} عزیز\nسفارش #{order_id} در وضعیت {status_text} است.\nمبلغ: {total}\nبا تشکر از {site_name}',
        'sms_statuses'  => array( 'completed' ),
    );

    public function __construct() {
        register_activation_hook( __FILE__, array( $this, 'on_activate' ) );

        add_action( 'admin_menu', array( $this, 'add_settings_page' ) );
        add_action( 'admin_init', array( $this, 'register_settings' ) );

        // استفاده از hook تغییر وضعیت برای پیام‌های مختلف
        add_action( 'woocommerce_order_status_changed', array( $this, 'order_status_changed_handler' ), 10, 3 );
        
        // اضافه کردن hook برای سفارش‌های جدید (حتی قبل از پرداخت)
        add_action( 'woocommerce_new_order', array( $this, 'new_order_handler' ), 10, 1 );
    }

    public function on_activate() {
        $options = get_option( self::OPTION_KEY );
        if ( $options === false ) {
            update_option( self::OPTION_KEY, $this->defaults );
        } else {
            // به‌روزرسانی تنظیمات موجود با مقادیر پیش‌فرض جدید
            $updated_options = wp_parse_args( $options, $this->defaults );
            update_option( self::OPTION_KEY, $updated_options );
        }
    }

    public function get_options() {
        $opts = get_option( self::OPTION_KEY, array() );
        return wp_parse_args( $opts, $this->defaults );
    }

    /**
     * Check if message was already sent for this order and status
     */
    private function is_message_sent( $order_id, $status ) {
        $sent_messages = get_option( self::SENT_MESSAGES_KEY, array() );
        $key = $order_id . '_' . $status;
        return isset( $sent_messages[ $key ] );
    }

    /**
     * Mark message as sent for this order and status
     */
    private function mark_message_sent( $order_id, $status ) {
        $sent_messages = get_option( self::SENT_MESSAGES_KEY, array() );
        $key = $order_id . '_' . $status;
        $sent_messages[ $key ] = current_time( 'mysql' );
        update_option( self::SENT_MESSAGES_KEY, $sent_messages );
    }

    /* ---------- Admin Settings ---------- */
    public function add_settings_page() {
        add_submenu_page(
            'woocommerce',
            'اطلاع رسانی آفیس باز',
            'اطلاع رسانی آفیس باز',
            'manage_woocommerce',
            'wc-whatsapp-sender',
            array( $this, 'settings_page_html' )
        );
    }

    public function register_settings() {
        register_setting( 'wc_whatsapp_sender_group', self::OPTION_KEY, array( $this, 'validate_options' ) );
    }

    public function validate_options( $input ) {
        $out = array();
        $out['api_url']      = isset( $input['api_url'] ) ? esc_url_raw( trim( $input['api_url'] ) ) : $this->defaults['api_url'];
        $out['api_key']      = isset( $input['api_key'] ) ? sanitize_text_field( $input['api_key'] ) : '';
        $out['sender']       = isset( $input['sender'] ) ? sanitize_text_field( $input['sender'] ) : '';
        $out['phone_field']  = isset( $input['phone_field'] ) ? sanitize_text_field( $input['phone_field'] ) : $this->defaults['phone_field'];
        $out['message']      = isset( $input['message'] ) ? wp_kses_post( $input['message'] ) : $this->defaults['message'];
        $out['status_messages'] = isset( $input['status_messages'] ) ? array_map( 'wp_kses_post', $input['status_messages'] ) : $this->defaults['status_messages'];
        $out['enabled']      = ( isset( $input['enabled'] ) && $input['enabled'] === 'yes' ) ? 'yes' : 'no';
        $out['log_to_order'] = ( isset( $input['log_to_order'] ) && $input['log_to_order'] === 'yes' ) ? 'yes' : 'no';
        $out['add_country']  = isset( $input['add_country'] ) ? preg_replace('/\D+/', '', $input['add_country'] ) : $this->defaults['add_country'];
        $out['order_statuses'] = isset( $input['order_statuses'] ) ? array_map( 'sanitize_text_field', $input['order_statuses'] ) : $this->defaults['order_statuses'];
        $out['admin_notify'] = ( isset( $input['admin_notify'] ) && $input['admin_notify'] === 'yes' ) ? 'yes' : 'no';
        $out['admin_phone']  = isset( $input['admin_phone'] ) ? sanitize_textarea_field( $input['admin_phone'] ) : '';
        $out['admin_message'] = isset( $input['admin_message'] ) ? wp_kses_post( $input['admin_message'] ) : $this->defaults['admin_message'];
        $out['sms_enabled']  = ( isset( $input['sms_enabled'] ) && $input['sms_enabled'] === 'yes' ) ? 'yes' : 'no';
        $out['sms_username'] = isset( $input['sms_username'] ) ? sanitize_text_field( $input['sms_username'] ) : '';
        $out['sms_password'] = isset( $input['sms_password'] ) ? sanitize_text_field( $input['sms_password'] ) : '';
        $out['sms_from']     = isset( $input['sms_from'] ) ? sanitize_text_field( $input['sms_from'] ) : '';
        $out['sms_wsdl']     = isset( $input['sms_wsdl'] ) ? esc_url_raw( trim( $input['sms_wsdl'] ) ) : $this->defaults['sms_wsdl'];
        $out['sms_message']  = isset( $input['sms_message'] ) ? sanitize_textarea_field( $input['sms_message'] ) : $this->defaults['sms_message'];
        $out['sms_statuses'] = isset( $input['sms_statuses'] ) ? array_map( 'sanitize_text_field', (array) $input['sms_statuses'] ) : $this->defaults['sms_statuses'];
        return $out;
    }

    public function settings_page_html() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            return;
        }

        $opts = $this->get_options();
        $active_tab = 'whatsapp';
        
        // نمایش پیام تأیید ذخیره‌سازی
        if ( isset( $_GET['settings-updated'] ) && $_GET['settings-updated'] ) {
            echo '<div class="notice notice-success is-dismissible"><p>تنظیمات با موفقیت ذخیره شد.</p></div>';
        }
        ?>
        <div class="wrap">
            <h1>اطلاع رسانی آفیس باز</h1>
            <form method="post" action="options.php">
                <?php settings_fields( 'wc_whatsapp_sender_group' ); ?>
                <input type="hidden" name="wc_ws_active_tab" id="wc_ws_active_tab" value="<?php echo esc_attr( $active_tab ); ?>" />

                <h2 class="nav-tab-wrapper wc-ws-tabs">
                    <a href="#" class="nav-tab nav-tab-active" data-tab="whatsapp">تنظیمات واتساپ</a>
                    <a href="#" class="nav-tab" data-tab="sms">ارسال پیامک</a>
                    <a href="#" class="nav-tab" data-tab="guide">آموزش</a>
                </h2>

                <div class="wc-ws-tab-content" data-tab="whatsapp">

                <table class="form-table">
                    <tr>
                        <th><label for="wc_ws_api_url">API URL</label></th>
                        <td><input name="<?php echo self::OPTION_KEY; ?>[api_url]" type="text" id="wc_ws_api_url" value="<?php echo esc_attr( $opts['api_url'] ); ?>" class="regular-text" /></td>
                    </tr>
                    <tr>
                        <th><label for="wc_ws_api_key">API Key</label></th>
                        <td><input name="<?php echo self::OPTION_KEY; ?>[api_key]" type="text" id="wc_ws_api_key" value="<?php echo esc_attr( $opts['api_key'] ); ?>" class="regular-text" />
                        <p class="description">کلید API خود را وارد کنید (مثال: ZTHZ6RXB...)</p></td>
                    </tr>
                    <tr>
                        <th><label for="wc_ws_sender">Sender (فرستنده)</label></th>
                        <td><input name="<?php echo self::OPTION_KEY; ?>[sender]" type="text" id="wc_ws_sender" value="<?php echo esc_attr( $opts['sender'] ); ?>" class="regular-text" />
                        <p class="description">نام دستگاه را از پنل واتساپ آفیس بخش دستگاههای متصل وارد کنید </p></td>
                    </tr>
                    <tr>
                        <th><label for="wc_ws_add_country">کد کشور (پیشفرض برای ایران)</label></th>
                        <td><input name="<?php echo self::OPTION_KEY; ?>[add_country]" type="text" id="wc_ws_add_country" value="<?php echo esc_attr( $opts['add_country'] ); ?>" class="small-text" />
                        <p class="description">پیشفرض: <code>98</code> — اگر شماره‌ها بدون کد کشور هستند، این مقدار اضافه خواهد شد.</p></td>
                    </tr>
                    <tr>
                        <th><label for="wc_ws_phone_field">فیلد شماره از سفارش</label></th>
                        <td>
                            <select name="<?php echo self::OPTION_KEY; ?>[phone_field]" id="wc_ws_phone_field">
                                <option value="billing_phone" <?php selected( $opts['phone_field'], 'billing_phone' ); ?>>شماره صورتحساب (billing_phone)</option>
                                <option value="shipping_phone" <?php selected( $opts['phone_field'], 'shipping_phone' ); ?>>شماره ارسال (shipping_phone)</option>
                                <option value="phone" <?php selected( $opts['phone_field'], 'phone' ); ?>>شماره عمومی (phone)</option>
                                <option value="mobile" <?php selected( $opts['phone_field'], 'mobile' ); ?>>موبایل (mobile)</option>
                                <option value="auto_search" <?php selected( $opts['phone_field'], 'auto_search' ); ?>>جستجوی خودکار (پیشنهادی)</option>
                            </select>
                            <p class="description">
                                <strong>جستجوی خودکار:</strong> ابتدا از اطلاعات سفارش، سپس از اطلاعات کاربر جستجو می‌کند.<br>
                                فیلدهای جستجو: billing_phone, phone, mobile, user_phone, telephone
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="wc_ws_message">قالب پیام</label></th>
                        <td><textarea name="<?php echo self::OPTION_KEY; ?>[message]" id="wc_ws_message" class="large-text" rows="4"><?php echo esc_textarea( $opts['message'] ); ?></textarea>
                        <p class="description">می‌توانید از placeholder های زیر استفاده کنید:<br>
                        <code>{first_name}</code> - نام مشتری<br>
                        <code>{last_name}</code> - نام خانوادگی مشتری<br>
                        <code>{order_id}</code> - شماره سفارش<br>
                        <code>{total}</code> - مجموع سفارش<br>
                        <code>{site_name}</code> - نام سایت<br>
                        <code>{status_text}</code> - وضعیت سفارش (تکمیل شد، در حال انجام است، ...)<br>
                        <code>{status_message}</code> - پیام مخصوص وضعیت (با ایموجی و متن مناسب)<br>
                        <code>{products_list}</code> - لیست محصولات سفارش با تعداد و قیمت<br>
                        <code>{last_notes}</code> - دو یادداشت آخر سفارش (فقط برای وضعیت تکمیل شده)<br><br>
                        <strong>💡 پیشنهاد:</strong> می‌توانید ایموجی‌های مناسب اضافه کنید:<br>
                        👋 سلام، 🏪 فروشگاه، 📦 سفارش، 💰 قیمت، 📝 یادداشت</p></td>
                    </tr>
                </table>

                <h2>نمونه قالب‌های پیشنهادی</h2>
                <div style="background: #f0f8ff; padding: 15px; border-radius: 5px; margin-bottom: 20px;">
                    <p><strong>🎨 نمونه قالب‌های زیبا:</strong></p>
                    <div style="display: flex; gap: 15px; flex-wrap: wrap;">
                        <div style="flex: 1; min-width: 300px; background: white; padding: 15px; border: 1px solid #ddd; border-radius: 5px;">
                            <strong>قالب ساده:</strong><br>
                            <code style="display: block; background: #f9f9f9; padding: 10px; margin: 10px 0; border-radius: 3px; white-space: pre-line;">👋 سلام {first_name} عزیز
🏪 {site_name}
📦 سفارش #{order_id}
💰 مجموع: {total}

🛍️ محصولات سفارش:
{products_list}

{status_message}</code>
                            <button type="button" class="button button-secondary" onclick="selectTemplate('simple')">انتخاب این قالب</button>
                        </div>
                        <div style="flex: 1; min-width: 300px; background: white; padding: 15px; border: 1px solid #ddd; border-radius: 5px;">
                            <strong>قالب کامل:</strong><br>
                            <code style="display: block; background: #f9f9f9; padding: 10px; margin: 10px 0; border-radius: 3px; white-space: pre-line;">👋 سلام {first_name} عزیز
🏪 {site_name}
📦 سفارش #{order_id}
💰 مجموع: {total}

🛍️ محصولات سفارش:
{products_list}

{status_message}
{last_notes}</code>
                            <button type="button" class="button button-secondary" onclick="selectTemplate('complete')">انتخاب این قالب</button>
                        </div>
                        <div style="flex: 1; min-width: 300px; background: white; padding: 15px; border: 1px solid #ddd; border-radius: 5px;">
                            <strong>قالب حرفه‌ای:</strong><br>
                            <code style="display: block; background: #f9f9f9; padding: 10px; margin: 10px 0; border-radius: 3px; white-space: pre-line;">🎉 سلام {first_name} عزیز
🏪 از {site_name} تشکر می‌کنیم
📦 سفارش شما: #{order_id}
💰 مبلغ: {total}

🛍️ محصولات سفارش:
{products_list}

{status_message}
📞 برای سوالات: تماس با پشتیبانی</code>
                            <button type="button" class="button button-secondary" onclick="selectTemplate('professional')">انتخاب این قالب</button>
                        </div>
                    </div>
                </div>

                <script>
                function selectTemplate(type) {
                    var templates = {
                        'simple': '👋 سلام {first_name} عزیز\n\n🏪 {site_name}\n\n📦 سفارش #{order_id}\n💰 مجموع: {total}\n\n🛍️ محصولات سفارش:\n{products_list}\n\n{status_message}',
                        'complete': '👋 سلام {first_name} عزیز\n\n🏪 {site_name}\n\n📦 سفارش #{order_id}\n💰 مجموع: {total}\n\n🛍️ محصولات سفارش:\n{products_list}\n\n{status_message}\n\n{last_notes}',
                        'professional': '🎉 سلام {first_name} عزیز\n\n🏪 از {site_name} تشکر می‌کنیم\n\n📦 سفارش شما: #{order_id}\n💰 مبلغ: {total}\n\n🛍️ محصولات سفارش:\n{products_list}\n\n{status_message}\n\n📞 برای سوالات: تماس با پشتیبانی'
                    };
                    
                    var messageField = document.getElementById('wc_ws_message');
                    if (messageField && templates[type]) {
                        messageField.value = templates[type];
                        messageField.style.border = '2px solid #4CAF50';
                        setTimeout(function() {
                            messageField.style.border = '';
                        }, 2000);
                    }
                }
                </script>

                <h2>پیام‌های مخصوص هر وضعیت</h2>
                <div style="background: #e7f3ff; padding: 15px; border-radius: 5px; margin-bottom: 20px;">
                    <p><strong>💡 راهنمای ایموجی‌های پیشنهادی:</strong></p>
                    <ul style="list-style: none; padding: 0;">
                        <li style="margin: 5px 0;">⏳ برای انتظار پرداخت</li>
                        <li style="margin: 5px 0;">🔄 برای در حال انجام</li>
                        <li style="margin: 5px 0;">✅ برای تکمیل شده</li>
                        <li style="margin: 5px 0;">⏸️ برای در انتظار</li>
                        <li style="margin: 5px 0;">❌ برای لغو شده</li>
                        <li style="margin: 5px 0;">💰 برای بازپرداخت</li>
                        <li style="margin: 5px 0;">⚠️ برای ناموفق</li>
                    </ul>
                </div>
                <table class="form-table">
                    <tr>
                        <th><label for="wc_ws_status_pending">پیام وضعیت "در انتظار پرداخت"</label></th>
                        <td><textarea name="<?php echo self::OPTION_KEY; ?>[status_messages][pending]" id="wc_ws_status_pending" class="large-text" rows="2"><?php echo esc_textarea( $opts['status_messages']['pending'] ); ?></textarea></td>
                    </tr>
                    <tr>
                        <th><label for="wc_ws_status_processing">پیام وضعیت "در حال انجام"</label></th>
                        <td><textarea name="<?php echo self::OPTION_KEY; ?>[status_messages][processing]" id="wc_ws_status_processing" class="large-text" rows="2"><?php echo esc_textarea( $opts['status_messages']['processing'] ); ?></textarea></td>
                    </tr>
                    <tr>
                        <th><label for="wc_ws_status_completed">پیام وضعیت "تکمیل شده"</label></th>
                        <td><textarea name="<?php echo self::OPTION_KEY; ?>[status_messages][completed]" id="wc_ws_status_completed" class="large-text" rows="2"><?php echo esc_textarea( $opts['status_messages']['completed'] ); ?></textarea></td>
                    </tr>
                    <tr>
                        <th><label for="wc_ws_status_onhold">پیام وضعیت "در انتظار"</label></th>
                        <td><textarea name="<?php echo self::OPTION_KEY; ?>[status_messages][on-hold]" id="wc_ws_status_onhold" class="large-text" rows="2"><?php echo esc_textarea( $opts['status_messages']['on-hold'] ); ?></textarea></td>
                    </tr>
                    <tr>
                        <th><label for="wc_ws_status_cancelled">پیام وضعیت "لغو شده"</label></th>
                        <td><textarea name="<?php echo self::OPTION_KEY; ?>[status_messages][cancelled]" id="wc_ws_status_cancelled" class="large-text" rows="2"><?php echo esc_textarea( $opts['status_messages']['cancelled'] ); ?></textarea></td>
                    </tr>
                    <tr>
                        <th><label for="wc_ws_status_refunded">پیام وضعیت "بازپرداخت شده"</label></th>
                        <td><textarea name="<?php echo self::OPTION_KEY; ?>[status_messages][refunded]" id="wc_ws_status_refunded" class="large-text" rows="2"><?php echo esc_textarea( $opts['status_messages']['refunded'] ); ?></textarea></td>
                    </tr>
                    <tr>
                        <th><label for="wc_ws_status_failed">پیام وضعیت "ناموفق"</label></th>
                        <td><textarea name="<?php echo self::OPTION_KEY; ?>[status_messages][failed]" id="wc_ws_status_failed" class="large-text" rows="2"><?php echo esc_textarea( $opts['status_messages']['failed'] ); ?></textarea></td>
                    </tr>
                </table>

                <h2>تنظیمات ارسال پیام</h2>
                <table class="form-table">
                    <tr>
                        <th><label for="wc_ws_order_status">وضعیت سفارش برای ارسال پیام</label></th>
                        <td>
                            <fieldset>
                                <label><input type="checkbox" name="<?php echo self::OPTION_KEY; ?>[order_statuses][]" value="pending" <?php checked( in_array( 'pending', (array) $opts['order_statuses'] ) ); ?> /> در انتظار پرداخت (Pending)</label><br>
                                <label><input type="checkbox" name="<?php echo self::OPTION_KEY; ?>[order_statuses][]" value="processing" <?php checked( in_array( 'processing', (array) $opts['order_statuses'] ) ); ?> /> در حال انجام (Processing)</label><br>
                                <label><input type="checkbox" name="<?php echo self::OPTION_KEY; ?>[order_statuses][]" value="completed" <?php checked( in_array( 'completed', (array) $opts['order_statuses'] ) ); ?> /> تکمیل شده (Completed)</label><br>
                                <label><input type="checkbox" name="<?php echo self::OPTION_KEY; ?>[order_statuses][]" value="on-hold" <?php checked( in_array( 'on-hold', (array) $opts['order_statuses'] ) ); ?> /> در انتظار (On Hold)</label><br>
                                <label><input type="checkbox" name="<?php echo self::OPTION_KEY; ?>[order_statuses][]" value="cancelled" <?php checked( in_array( 'cancelled', (array) $opts['order_statuses'] ) ); ?> /> لغو شده (Cancelled)</label><br>
                                <label><input type="checkbox" name="<?php echo self::OPTION_KEY; ?>[order_statuses][]" value="refunded" <?php checked( in_array( 'refunded', (array) $opts['order_statuses'] ) ); ?> /> بازپرداخت شده (Refunded)</label><br>
                                <label><input type="checkbox" name="<?php echo self::OPTION_KEY; ?>[order_statuses][]" value="failed" <?php checked( in_array( 'failed', (array) $opts['order_statuses'] ) ); ?> /> ناموفق (Failed)</label>
                            </fieldset>
                            <p class="description">
                                <strong>راهنمای انتخاب وضعیت‌ها:</strong><br>
                                • <strong>در انتظار پرداخت:</strong> وقتی سفارش ثبت می‌شود اما پرداخت نشده<br>
                                • <strong>در حال انجام:</strong> وقتی سفارش پرداخت شده و در حال آماده‌سازی است<br>
                                • <strong>تکمیل شده:</strong> وقتی سفارش آماده ارسال است<br>
                                • <strong>در انتظار:</strong> وقتی سفارش نیاز به بررسی دارد<br>
                                • <strong>لغو شده:</strong> وقتی سفارش لغو می‌شود<br>
                                • <strong>بازپرداخت شده:</strong> وقتی مبلغ بازپرداخت می‌شود<br>
                                • <strong>ناموفق:</strong> وقتی مشکلی در پردازش سفارش رخ می‌دهد
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th>فعالسازی</th>
                        <td><label><input type="checkbox" name="<?php echo self::OPTION_KEY; ?>[enabled]" value="yes" <?php checked( $opts['enabled'], 'yes' ); ?> /> فعال</label></td>
                    </tr>
                </table>

                <h2>اعلان مدیر</h2>
                <table class="form-table">
                    <tr>
                        <th>ارسال اعلان به مدیر</th>
                        <td><label><input type="checkbox" name="<?php echo self::OPTION_KEY; ?>[admin_notify]" value="yes" <?php checked( $opts['admin_notify'], 'yes' ); ?> /> فعال</label>
                        <p class="description">در صورت فعال بودن، پیام اعلان به شماره مدیر ارسال می‌شود.</p></td>
                    </tr>
                    <tr>
                        <th><label for="wc_ws_admin_phone">شماره‌های مدیر</label></th>
                        <td>
                            <textarea name="<?php echo self::OPTION_KEY; ?>[admin_phone]" id="wc_ws_admin_phone" class="large-text" rows="3"><?php echo esc_textarea( $opts['admin_phone'] ); ?></textarea>
                            <p class="description">هر شماره را در یک خط جدا یا با کاما وارد کنید (مثال: 09123456789 یا +989123456789). برای ارسال پیام به چند مدیر، شماره‌ها را پشت سر هم بنویسید.</p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="wc_ws_admin_message">قالب پیام مدیر</label></th>
                        <td><textarea name="<?php echo self::OPTION_KEY; ?>[admin_message]" id="wc_ws_admin_message" class="large-text" rows="8"><?php echo esc_textarea( $opts['admin_message'] ); ?></textarea>
                        <p class="description">می‌توانید از placeholder های زیر استفاده کنید:<br>
                        <code>{order_id}</code> - شماره سفارش<br>
                        <code>{customer_name}</code> - نام مشتری<br>
                        <code>{customer_phone}</code> - تلفن مشتری<br>
                        <code>{customer_email}</code> - ایمیل مشتری<br>
                        <code>{shipping_address}</code> - آدرس ارسال<br>
                        <code>{products_list}</code> - لیست محصولات<br>
                        <code>{total}</code> - مجموع سفارش<br>
                        <code>{order_date}</code> - تاریخ سفارش<br>
                        <code>{status_text}</code> - وضعیت سفارش<br>
                        <code>{status_message}</code> - پیام وضعیت</p></td>
                    </tr>
                    <tr>
                        <th>نوشتن لاگ در یادداشت سفارش</th>
                        <td><label><input type="checkbox" name="<?php echo self::OPTION_KEY; ?>[log_to_order]" value="yes" <?php checked( $opts['log_to_order'], 'yes' ); ?> /> بله</label>
                        <p class="description">در صورت فعال بودن، پاسخ یا خطاهای API به‌عنوان یادداشت خصوصی به سفارش اضافه می‌شوند.</p></td>
                    </tr>
                </table>

                </div>

                <div class="wc-ws-tab-content" data-tab="sms" style="display: none;">
                    <h2>تنظیمات پیامک (ملی پیامک)</h2>
                    <table class="form-table">
                        <tr>
                            <th>فعالسازی پیامک</th>
                            <td>
                                <label>
                                    <input type="checkbox" name="<?php echo self::OPTION_KEY; ?>[sms_enabled]" value="yes" <?php checked( $opts['sms_enabled'], 'yes' ); ?> />
                                    فعال
                                </label>
                                <p class="description">برای ارسال خودکار پیامک به مشتریان از طریق سرویس ملی پیامک این گزینه را فعال کنید.</p>
                            </td>
                        </tr>
                        <tr>
                            <th><label for="wc_ws_sms_username">نام کاربری</label></th>
                            <td><input name="<?php echo self::OPTION_KEY; ?>[sms_username]" type="text" id="wc_ws_sms_username" value="<?php echo esc_attr( $opts['sms_username'] ); ?>" class="regular-text" /></td>
                        </tr>
                        <tr>
                            <th><label for="wc_ws_sms_password">رمز عبور</label></th>
                            <td><input name="<?php echo self::OPTION_KEY; ?>[sms_password]" type="password" id="wc_ws_sms_password" value="<?php echo esc_attr( $opts['sms_password'] ); ?>" class="regular-text" autocomplete="new-password" /></td>
                        </tr>
                        <tr>
                            <th><label for="wc_ws_sms_from">شماره ارسال‌کننده</label></th>
                            <td><input name="<?php echo self::OPTION_KEY; ?>[sms_from]" type="text" id="wc_ws_sms_from" value="<?php echo esc_attr( $opts['sms_from'] ); ?>" class="regular-text" />
                            <p class="description">شماره اختصاصی پنل (مثال: 5000...)</p></td>
                        </tr>
                        <tr>
                            <th><label for="wc_ws_sms_wsdl">آدرس سرویس (WSDL)</label></th>
                            <td><input name="<?php echo self::OPTION_KEY; ?>[sms_wsdl]" type="text" id="wc_ws_sms_wsdl" value="<?php echo esc_attr( $opts['sms_wsdl'] ); ?>" class="regular-text" />
                            <p class="description">در صورت نیاز می‌توانید آدرس وب‌سرویس ملی پیامک را تغییر دهید.</p></td>
                        </tr>
                        <tr>
                            <th><label for="wc_ws_sms_message">قالب پیامک</label></th>
                            <td>
                                <textarea name="<?php echo self::OPTION_KEY; ?>[sms_message]" id="wc_ws_sms_message" class="large-text" rows="6"><?php echo esc_textarea( $opts['sms_message'] ); ?></textarea>
                                <p class="description">Placeholder های قابل استفاده: <code>{first_name}</code>، <code>{last_name}</code>، <code>{order_id}</code>، <code>{total}</code>، <code>{site_name}</code>، <code>{status_text}</code>.</p>
                            </td>
                        </tr>
                        <tr>
                            <th>وضعیت‌های ارسال پیامک</th>
                            <td>
                                <fieldset>
                                    <?php
                                    $statuses = array(
                                        'pending'    => 'در انتظار پرداخت',
                                        'processing' => 'در حال انجام',
                                        'completed'  => 'تکمیل شده',
                                        'on-hold'    => 'در انتظار',
                                        'cancelled'  => 'لغو شده',
                                        'refunded'   => 'بازپرداخت شده',
                                        'failed'     => 'ناموفق',
                                    );
                                    foreach ( $statuses as $status_key => $status_label ) :
                                    ?>
                                        <label>
                                            <input type="checkbox" name="<?php echo self::OPTION_KEY; ?>[sms_statuses][]" value="<?php echo esc_attr( $status_key ); ?>" <?php checked( in_array( $status_key, (array) $opts['sms_statuses'], true ) ); ?> />
                                            <?php echo esc_html( $status_label ); ?>
                                        </label>
                                        <br>
                                    <?php endforeach; ?>
                                </fieldset>
                                <p class="description">پیامک فقط برای وضعیت‌های انتخاب‌شده ارسال می‌شود.</p>
                            </td>
                        </tr>
                    </table>
                </div>

                <div class="wc-ws-tab-content" data-tab="guide" style="display: none;">
                    <h2>آموزش گام‌به‌گام</h2>
                    <ol style="line-height: 1.9;">
                        <li>افزونه را فعال کنید و از تب «تنظیمات واتساپ» کلید API و نام دستگاه متصل در پنل آفیس باز را وارد نمایید.</li>
                        <li>وضعیت‌های سفارش مورد نیاز را تیک بزنید و متن پیام مشتری و مدیر را مطابق کسب‌وکار خود شخصی‌سازی کنید.</li>
                        <li>در صورت نیاز به ارسال پیامک، به تب «ارسال پیامک» بروید، گزینه فعال‌سازی را روشن و اطلاعات پنل ملی پیامک (نام کاربری، رمز، خط اختصاصی) را تکمیل کنید.</li>
                        <li>از بخش «تست دستی» شماره خود را وارد کنید و دکمه تست مربوطه را بزنید تا همه تنظیمات قبل از سفارش واقعی بررسی شود.</li>
                        <li>پس از تایید تست، سفارش واقعی ثبت کنید یا وضعیت سفارشات را تغییر دهید تا پیام‌ها برای مشتری و مدیران ارسال شوند.</li>
                        <li>در صورت مشاهده خطا، به یادداشت‌های سفارش مراجعه و متن خطا را بررسی کنید؛ معمولاً مشکل از شماره، تنظیمات API یا محدودیت پنل است.</li>
                    </ol>
                    <div style="margin-top: 20px; display: flex; gap: 10px; flex-wrap: wrap;">
                        <a href="https://web.officebaz.ir/docs/?article=woo-plugin" class="button button-primary" target="_blank">مشاهده ویدیو آموزشی</a>
                        <a href="https://web.officebaz.ir/docs/?article=woo-plugin" class="button" target="_blank">دانلود آخرین آپدیت</a>
                    </div>
                </div>

                <?php submit_button(); ?>
            </form>

            <script>
            (function() {
                const navTabs = document.querySelectorAll('.wc-ws-tabs .nav-tab');
                const tabContents = document.querySelectorAll('.wc-ws-tab-content');
                const activeInput = document.getElementById('wc_ws_active_tab');
                const validTabs = Array.from(navTabs).map(function(link) { return link.dataset.tab; });

                function activateTab(tab) {
                    if ( validTabs.indexOf(tab) === -1 ) {
                        tab = 'whatsapp';
                    }
                    navTabs.forEach(function(link) {
                        link.classList.toggle('nav-tab-active', link.dataset.tab === tab);
                    });
                    tabContents.forEach(function(section) {
                        section.style.display = section.dataset.tab === tab ? '' : 'none';
                    });
                    if (activeInput) {
                        activeInput.value = tab;
                    }
                }

                navTabs.forEach(function(link) {
                    link.addEventListener('click', function(e) {
                        e.preventDefault();
                        activateTab(link.dataset.tab);
                        window.location.hash = link.dataset.tab;
                    });
                });

                const initialTab = window.location.hash ? window.location.hash.replace('#', '') : ( activeInput ? activeInput.value : 'whatsapp' );
                activateTab(initialTab);
            })();
            </script>
            <h2>تست دستی</h2>
            <p>برای ارسال تست یک پیام با پارامترهای کنونی:</p>
            <form method="post">
                <?php wp_nonce_field( 'wc_ws_test_send', 'wc_ws_test_nonce' ); ?>
                <table class="form-table">
                    <tr>
                        <th><label for="wc_ws_test_number">شماره تلفن</label></th>
                        <td><input type="text" name="wc_ws_test_number" id="wc_ws_test_number" placeholder="مثال: 09123456789 یا +989123456789 یا 9123456789" class="regular-text" /></td>
                    </tr>
                    <tr>
                        <th><label for="wc_ws_test_status">وضعیت تست</label></th>
                        <td>
                            <select name="wc_ws_test_status" id="wc_ws_test_status">
                                <option value="pending">در انتظار پرداخت</option>
                                <option value="processing">در حال انجام</option>
                                <option value="completed">تکمیل شده</option>
                                <option value="on-hold">در انتظار</option>
                                <option value="cancelled">لغو شده</option>
                                <option value="refunded">بازپرداخت شده</option>
                                <option value="failed">ناموفق</option>
                            </select>
                        </td>
                    </tr>
                </table>
                <p>
                    <input type="submit" name="wc_ws_test_whatsapp" class="button button-primary" value="ارسال تست واتساپ" />
                    <input type="submit" name="wc_ws_test_sms" class="button" value="ارسال تست پیامک" />
                </p>
                <p class="description">هر دکمه کانال مربوط به خود را تست می‌کند.</p>
            </form>
            <?php
            $is_whatsapp_test = isset( $_POST['wc_ws_test_whatsapp'] );
            $is_sms_test = isset( $_POST['wc_ws_test_sms'] );

            if ( ( $is_whatsapp_test || $is_sms_test ) && check_admin_referer( 'wc_ws_test_send', 'wc_ws_test_nonce' ) ) {
                $number = isset( $_POST['wc_ws_test_number'] ) ? wp_unslash( $_POST['wc_ws_test_number'] ) : '';
                $test_status = isset( $_POST['wc_ws_test_status'] ) ? sanitize_text_field( $_POST['wc_ws_test_status'] ) : 'completed';
                $normalized = $this->normalize_iran_phone( $number, $opts['add_country'] );
                
                if ( $normalized ) {
                    // استفاده از قالب پیام فعلی
                    $test_placeholders = array(
                        '{first_name}' => 'تست',
                        '{last_name}'  => 'کاربر',
                        '{order_id}'   => '12345',
                        '{total}'      => '150,000 تومان',
                        '{site_name}'  => get_bloginfo( 'name' ),
                        '{status_text}' => $this->get_status_text( $test_status ),
                        '{status_message}' => $this->get_status_message( $test_status ),
                        '{products_list}' => "• محصول تستی 1 (تعداد: 2) - 50,000 تومان\n• محصول تستی 2 (تعداد: 1) - 100,000 تومان",
                        '{last_notes}' => "\n\n📝 یادداشت تستی 1: سفارش آماده ارسال است\n📝 یادداشت تستی 2: کد رهگیری: 1234567890",
                    );
                    
                    $test_message = strtr( $opts['message'], $test_placeholders );
                    $test_message = wp_strip_all_tags( $test_message );
                    $test_message = html_entity_decode( $test_message, ENT_QUOTES, 'UTF-8' );
                    
                    if ( $is_whatsapp_test ) {
                        $res = $this->send_message( $normalized, $test_message, $opts );
                        if ( is_wp_error( $res ) ) {
                            echo '<div class="error"><p>خطا در تست واتساپ: ' . esc_html( $res->get_error_message() ) . '</p></div>';
                        } else {
                            echo '<div class="updated"><p>پیام تست واتساپ ارسال شد به شماره: ' . esc_html( $normalized ) . '. پاسخ API: ' . esc_html( (string) $res ) . '</p></div>';
                            echo '<div class="notice notice-info"><p><strong>پیام ارسالی:</strong><br><pre>' . esc_html( $test_message ) . '</pre></p></div>';
                        }
                    }

                    if ( $is_sms_test ) {
                        $local_number = $this->format_phone_for_sms( $normalized );
                        if ( ! $local_number ) {
                            echo '<div class="error"><p>خطا در تست پیامک: شماره برای ارسال پیامک معتبر نیست.</p></div>';
                        } elseif ( empty( $opts['sms_username'] ) || empty( $opts['sms_password'] ) || empty( $opts['sms_from'] ) ) {
                            echo '<div class="error"><p>خطا در تست پیامک: اطلاعات کاربری یا خط ارسال‌کننده ناقص است.</p></div>';
                        } else {
                            $sms_params = array(
                                'username' => $opts['sms_username'],
                                'password' => $opts['sms_password'],
                                'to'       => $local_number,
                                'from'     => $opts['sms_from'],
                                'text'     => $test_message,
                                'isflash'  => false,
                            );
                            $sms_result = $this->send_sms_via_melipayamak( $sms_params, $opts['sms_wsdl'] );
                            if ( is_wp_error( $sms_result ) ) {
                                echo '<div class="error"><p>خطا در تست پیامک: ' . esc_html( $sms_result->get_error_message() ) . '</p></div>';
                            } else {
                                echo '<div class="updated"><p>پیامک تستی ارسال شد به شماره: ' . esc_html( $local_number ) . '. پاسخ سرویس: ' . esc_html( (string) $sms_result ) . '</p></div>';
                                echo '<div class="notice notice-info"><p><strong>متن پیامک:</strong><br><pre>' . esc_html( $test_message ) . '</pre></p></div>';
                            }
                        }
                    }
                } else {
                    echo '<div class="error"><p>شماره معتبر نیست. شماره نرمالایز شده: ' . esc_html( $normalized ) . '</p></div>';
                }
            }
            ?>


            <h2>راهنمای کد رهگیری</h2>
            <div style="background: #e7f3ff; padding: 15px; border-radius: 5px; margin-bottom: 20px;">
                <p><strong>📝 برای ارسال کد رهگیری:</strong></p>
                <ol>
                    <li>به سفارش بروید</li>
                    <li>در بخش "Order Notes" یادداشت جدید اضافه کنید</li>
                    <li>متن: <code>کد رهگیری: 1234567890</code></li>
                </ol>
                <p><em>دو یادداشت آخر سفارش به صورت خودکار ارسال می‌شود.</em></p>
            </div>

            <h2>لینک‌های مفید</h2>
            <div style="background: #f0f8ff; padding: 15px; border-radius: 5px; margin-bottom: 20px;">
                <p><strong>🔗 لینک‌های مهم:</strong></p>
                <ul style="list-style: none; padding: 0;">
                    <li style="margin: 10px 0;">
                        <a href="https://web.officebaz.ir/docs/?article=woo-plugin" target="_blank" style="text-decoration: none; color: #0073aa; font-weight: bold;">
                            📚 مشاهده آموزش اطلاع رسانی آفیس باز
                        </a>
                    </li>
                    <li style="margin: 10px 0;">
                        <a href="https://web.officebaz.ir/" target="_blank" style="text-decoration: none; color: #0073aa; font-weight: bold;">
                            🔑 دریافت API اطلاع رسانی آفیس باز
                        </a>
                    </li>
                </ul>
            </div>

        </div>
        <?php
    }

    /* ---------- Order handler ---------- */
    public function order_status_changed_handler( $order_id, $old_status, $new_status ) {
        error_log( '[WC WhatsApp] Order status changed: ' . $order_id . ' from ' . $old_status . ' to ' . $new_status );
        
        // فقط اگر وضعیت واقعاً تغییر کرده باشد
        if ( $old_status !== $new_status ) {
            $this->order_status_handler( $order_id, $new_status );
        }
    }

    public function new_order_handler( $order_id ) {
        error_log( '[WC WhatsApp] New order created: ' . $order_id );
        
        // برای سفارش جدید، وضعیت pending را بررسی کن
        $this->order_status_handler( $order_id, 'pending' );
    }

    public function order_status_handler( $order_id, $current_status = null ) {
        if ( ! function_exists( 'wc_get_order' ) ) {
            error_log( '[WC WhatsApp] WooCommerce not available' );
            return;
        }

        $opts = $this->get_options();
        if ( $opts['enabled'] !== 'yes' ) {
            error_log( '[WC WhatsApp] Plugin disabled' );
            return;
        }

        $order = wc_get_order( $order_id );
        if ( ! $order ) {
            error_log( '[WC WhatsApp] Order not found: ' . $order_id );
            return;
        }

        // اگر وضعیت مشخص نشده، از سفارش بگیر
        if ( $current_status === null ) {
            $current_status = $order->get_status();
        }
        
        $selected_statuses = $opts['order_statuses'];

        error_log( '[WC WhatsApp] Order ' . $order_id . ' status changed to: ' . $current_status . ', selected statuses: ' . implode( ', ', $selected_statuses ) );

        // بررسی اینکه آیا پیام قبلاً برای این سفارش و وضعیت ارسال شده یا نه
        if ( $this->is_message_sent( $order_id, $current_status ) ) {
            error_log( '[WC WhatsApp] Message already sent for order ' . $order_id . ' with status ' . $current_status . ', skipping...' );
            return;
        }

        // بررسی وضعیت سفارش
        $allowed_statuses = array( 'completed', 'processing', 'pending', 'on-hold', 'cancelled', 'refunded', 'failed' );
        
        // فقط برای وضعیت‌های مجاز پیام ارسال کن
        if ( ! in_array( $current_status, $allowed_statuses ) ) {
            error_log( '[WC WhatsApp] Status not allowed for messaging: ' . $current_status );
            return;
        }
        
        // بررسی اینکه آیا باید برای این وضعیت پیام ارسال شود
        if ( ! in_array( $current_status, $selected_statuses ) ) {
            error_log( '[WC WhatsApp] Status not selected for messaging. Current: ' . $current_status . ', Selected: ' . implode( ', ', $selected_statuses ) );
            return;
        }

        // خواندن شماره از فیلد مشخص شده
        $phone_raw = $this->get_customer_phone( $order, $opts );

        error_log( '[WC WhatsApp] Raw phone from order: ' . $phone_raw . ' (field: ' . $opts['phone_field'] . ')' );

        $phone = $this->normalize_iran_phone( $phone_raw, $opts['add_country'] );
        error_log( '[WC WhatsApp] Normalized phone: ' . $phone );
        
        if ( ! $phone ) {
            $this->log_order_note( $order, 'WhatsApp: شماره تلفن برای این سفارش یافت نشد یا قابل نرمال شدن نبود. ورودی: ' . $phone_raw );
            error_log( '[WC WhatsApp] Phone normalization failed for: ' . $phone_raw );
            return;
        }

        // جایگزینی placeholderها
        $last_notes = '';
        if ( $current_status === 'completed' ) {
            error_log( '[WC WhatsApp] Getting last notes for completed order: ' . $order->get_id() );
            $last_notes = $this->get_last_two_notes( $order );
            error_log( '[WC WhatsApp] Last notes result: "' . $last_notes . '"' );
            if ( ! empty( $last_notes ) ) {
                $last_notes = "\n\n" . $last_notes;
                error_log( '[WC WhatsApp] Last notes formatted: "' . $last_notes . '"' );
            } else {
                error_log( '[WC WhatsApp] No notes found for order: ' . $order->get_id() );
            }
        } else {
            error_log( '[WC WhatsApp] Order status is not completed: ' . $current_status );
        }

        $placeholders = array(
            '{order_id}'     => $order->get_id(),
            '{first_name}'   => $order->get_billing_first_name(),
            '{last_name}'    => $order->get_billing_last_name(),
            '{total}'        => $this->get_clean_order_total( $order ),
            '{site_name}'    => get_bloginfo( 'name' ),
            '{status_text}'  => $this->get_status_text( $current_status ),
            '{status_message}' => $this->get_status_message( $current_status ),
            '{products_list}' => $this->get_products_list( $order ),
            '{last_notes}'   => $last_notes,
        );

        $message = strtr( $opts['message'], $placeholders );

        // پاکسازی HTML tags
        $message = wp_strip_all_tags( $message );
        $message = html_entity_decode( $message, ENT_QUOTES, 'UTF-8' );
        
        error_log( '[WC WhatsApp] Final message: ' . $message );

        // ارسال پیامک در صورت فعال بودن
        $sms_sent = $this->send_sms_notification( $order, $current_status, $opts, $placeholders, $phone );

        // ارسال پیام واتساپ
        $whatsapp_ready = ! empty( $opts['api_key'] ) && ! empty( $opts['sender'] );
        $result = null;

        if ( $whatsapp_ready ) {
            $result = $this->send_message( $phone, $message, $opts );
        }
        
        // ارسال اعلان به مدیر
        $this->send_admin_notification( $order, $current_status, $opts );

        $whatsapp_sent = false;

        if ( $whatsapp_ready ) {
            if ( is_wp_error( $result ) ) {
                $this->log_order_note( $order, 'WhatsApp: خطا هنگام ارسال پیام (وضعیت: ' . $current_status . ') - ' . $result->get_error_message() );
            } else {
                $this->log_order_note( $order, 'WhatsApp: پیام ارسال شد به ' . $phone . ' (وضعیت: ' . $current_status . '). پاسخ API: ' . $result );
                $whatsapp_sent = true;
            }
        } else {
            $this->log_order_note( $order, 'WhatsApp: تنظیمات API ناقص است، پیام ارسال نشد.' );
        }

        $this->mark_message_sent( $order_id, $current_status );
    }

    /**
     * Get clean order total without HTML
     *
     * @param WC_Order $order
     * @return string
     */
    private function get_clean_order_total( $order ) {
        $total = $order->get_total();
        $currency = $order->get_currency();
        $symbol = get_woocommerce_currency_symbol( $currency );
        
        // اگر مجموع صفر است، از قیمت محصولات استفاده کن
        if ( $total == 0 ) {
            $total = 0;
            foreach ( $order->get_items() as $item ) {
                $total += $item->get_total();
            }
        }
        
        return number_format( $total ) . ' ' . $symbol;
    }

    /**
     * Get status text in Persian
     *
     * @param string $status
     * @return string
     */
    private function get_status_text( $status ) {
        $status_texts = array(
            'completed'  => 'تکمیل شد',
            'processing' => 'در حال انجام است',
            'pending'    => 'در انتظار پرداخت است',
            'on-hold'    => 'در انتظار است',
            'cancelled'  => 'لغو شد',
            'refunded'   => 'بازپرداخت شد',
            'failed'     => 'ناموفق بود',
        );
        
        return isset( $status_texts[ $status ] ) ? $status_texts[ $status ] : $status;
    }

    /**
     * Get status-specific message
     *
     * @param string $status
     * @return string
     */
    private function get_status_message( $status ) {
        $opts = $this->get_options();
        $status_messages = $opts['status_messages'];
        
        return isset( $status_messages[ $status ] ) ? $status_messages[ $status ] : 'وضعیت سفارش: ' . $status;
    }

    /**
     * Get first product image from order
     *
     * @param WC_Order $order
     * @return string|false
     */
    private function get_first_product_image( $order ) {
        foreach ( $order->get_items() as $item ) {
            $product = $item->get_product();
            if ( $product ) {
                // ابتدا تصویر شاخص محصول را بررسی کن
                $image_id = $product->get_image_id();
                if ( $image_id ) {
                    $image_url = wp_get_attachment_image_url( $image_id, 'large' );
                    if ( $image_url ) {
                        error_log( '[WC WhatsApp] Product featured image found: ' . $image_url );
                        return $image_url;
                    }
                }
                
                // اگر تصویر شاخص نبود، از گالری تصاویر استفاده کن
                $gallery_ids = $product->get_gallery_image_ids();
                if ( ! empty( $gallery_ids ) ) {
                    $image_url = wp_get_attachment_image_url( $gallery_ids[0], 'large' );
                    if ( $image_url ) {
                        error_log( '[WC WhatsApp] Product gallery image found: ' . $image_url );
                        return $image_url;
                    }
                }
                
                // اگر محصول متغیر است، از تصویر والد استفاده کن
                if ( $product->is_type( 'variation' ) ) {
                    $parent_id = $product->get_parent_id();
                    if ( $parent_id ) {
                        $parent_product = wc_get_product( $parent_id );
                        if ( $parent_product ) {
                            $parent_image_id = $parent_product->get_image_id();
                            if ( $parent_image_id ) {
                                $image_url = wp_get_attachment_image_url( $parent_image_id, 'large' );
                                if ( $image_url ) {
                                    error_log( '[WC WhatsApp] Parent product image found: ' . $image_url );
                                    return $image_url;
                                }
                            }
                        }
                    }
                }
            }
        }
        error_log( '[WC WhatsApp] No product image found for order: ' . $order->get_id() );
        return false;
    }

    /**
     * Send image via API
     *
     * @param string $number
     * @param string $image_url
     * @param array $opts
     * @param string $caption
     * @return mixed
     */
    private function send_image( $number, $image_url, $opts = null, $caption = '' ) {
        $opts = $opts ? $opts : $this->get_options();

        // استفاده از POST request برای ارسال تصویر
        $data = array(
            'api_key' => $opts['api_key'],
            'sender'  => $opts['sender'],
            'number'  => $number,
            'type'    => 'image',
            'url'     => $image_url,
            'caption' => $caption ? $caption : 'تصویر محصول سفارش شما'
        );

        $args = array(
            'method'  => 'POST',
            'timeout' => 30,
            'headers' => array( 
                'Content-Type' => 'application/json',
                'Accept' => 'application/json' 
            ),
            'body' => json_encode( $data )
        );

        // لاگ کامل برای دیباگ
        error_log( '[WC WhatsApp] Image API URL: ' . $opts['api_url'] );
        error_log( '[WC WhatsApp] Image data: ' . json_encode( $data ) );

        $response = wp_remote_request( $opts['api_url'], $args );

        if ( is_wp_error( $response ) ) {
            error_log( '[WC WhatsApp] Image request error: ' . $response->get_error_message() );
            return $response;
        }

        $code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );

        error_log( '[WC WhatsApp] Image HTTP ' . $code . ' - ' . $body );
        error_log( '[WC WhatsApp] Image data sent: ' . json_encode( $data ) );

        if ( $code >= 200 && $code < 300 ) {
            return $body;
        }

        return new WP_Error( 'wc_whatsapp_image_error', sprintf( 'Image API returned status %s', $code ) );
    }

    /**
     * Get last order note
     *
     * @param WC_Order $order
     * @return string
     */
    private function get_last_order_note( $order ) {
        $notes = wc_get_order_notes( array(
            'order_id' => $order->get_id(),
            'limit'    => 1,
            'orderby'  => 'date_created',
            'order'    => 'DESC',
        ) );

        error_log( '[WC WhatsApp] Getting last note for order: ' . $order->get_id() );
        error_log( '[WC WhatsApp] Found ' . count( $notes ) . ' notes' );

        if ( ! empty( $notes ) ) {
            $note = $notes[0];
            error_log( '[WC WhatsApp] Last note content: ' . $note->content );
            return $note->content;
        }

        error_log( '[WC WhatsApp] No notes found for order: ' . $order->get_id() );
        return '';
    }

    /**
     * Get last two order notes
     *
     * @param WC_Order $order
     * @return string
     */
    private function get_last_two_notes( $order ) {
        error_log( '[WC WhatsApp] Getting last two notes for order: ' . $order->get_id() );
        
        $notes = wc_get_order_notes( array(
            'order_id' => $order->get_id(),
            'limit'    => 2,
            'orderby'  => 'date_created',
            'order'    => 'DESC',
        ) );

        error_log( '[WC WhatsApp] Found ' . count( $notes ) . ' notes for order' );

        if ( empty( $notes ) ) {
            error_log( '[WC WhatsApp] No notes found for order: ' . $order->get_id() );
            return '';
        }

        $notes_text = '';
        foreach ( $notes as $note ) {
            $content = $note->content;
            error_log( '[WC WhatsApp] Adding note: "' . $content . '"' );
            $notes_text .= "📝 " . $content . "\n";
        }

        $notes_text = trim( $notes_text );
        error_log( '[WC WhatsApp] Final notes text: "' . $notes_text . '"' );
        return $notes_text;
    }

    /**
     * Get order products list
     *
     * @param WC_Order $order
     * @return string
     */
    private function get_products_list( $order ) {
        $products = array();
        foreach ( $order->get_items() as $item ) {
            $product_name = $item->get_name();
            $quantity = $item->get_quantity();
            $total = $item->get_total();
            $products[] = "• {$product_name} (تعداد: {$quantity}) - " . number_format( $total ) . " تومان";
        }
        return implode( "\n", $products );
    }

    /**
     * Get formatted shipping address
     *
     * @param WC_Order $order
     * @return string
     */
    private function get_shipping_address( $order ) {
        $address = array();
        
        // ابتدا آدرس ارسال را بررسی کن
        $shipping_address_1 = $order->get_shipping_address_1();
        $shipping_city = $order->get_shipping_city();
        
        // اگر آدرس ارسال خالی است، از آدرس صورتحساب استفاده کن
        if ( empty( $shipping_address_1 ) && empty( $shipping_city ) ) {
            // استفاده از آدرس صورتحساب
            if ( $order->get_billing_first_name() || $order->get_billing_last_name() ) {
                $address[] = $order->get_billing_first_name() . ' ' . $order->get_billing_last_name();
            }
            
            if ( $order->get_billing_address_1() ) {
                $address[] = $order->get_billing_address_1();
            }
            
            if ( $order->get_billing_address_2() ) {
                $address[] = $order->get_billing_address_2();
            }
            
            if ( $order->get_billing_city() ) {
                $address[] = $order->get_billing_city();
            }
            
            if ( $order->get_billing_state() ) {
                $address[] = $order->get_billing_state();
            }
            
            if ( $order->get_billing_postcode() ) {
                $address[] = $order->get_billing_postcode();
            }
            
            if ( $order->get_billing_country() ) {
                $address[] = $order->get_billing_country();
            }
        } else {
            // استفاده از آدرس ارسال
            if ( $order->get_shipping_first_name() || $order->get_shipping_last_name() ) {
                $address[] = $order->get_shipping_first_name() . ' ' . $order->get_shipping_last_name();
            }
            
            if ( $order->get_shipping_address_1() ) {
                $address[] = $order->get_shipping_address_1();
            }
            
            if ( $order->get_shipping_address_2() ) {
                $address[] = $order->get_shipping_address_2();
            }
            
            if ( $order->get_shipping_city() ) {
                $address[] = $order->get_shipping_city();
            }
            
            if ( $order->get_shipping_state() ) {
                $address[] = $order->get_shipping_state();
            }
            
            if ( $order->get_shipping_postcode() ) {
                $address[] = $order->get_shipping_postcode();
            }
            
            if ( $order->get_shipping_country() ) {
                $address[] = $order->get_shipping_country();
            }
        }
        
        return implode( "\n", $address );
    }

    /**
     * Send admin notification
     *
     * @param WC_Order $order
     * @param string $current_status
     * @param array $opts
     * @return void
     */
    private function send_admin_notification( $order, $current_status, $opts ) {
        if ( $opts['admin_notify'] !== 'yes' || empty( $opts['admin_phone'] ) ) {
            return;
        }

        $raw_numbers = preg_split( '/[\s,]+/', $opts['admin_phone'] );
        $admin_numbers = array();

        foreach ( $raw_numbers as $raw_number ) {
            $raw_number = trim( $raw_number );
            if ( empty( $raw_number ) ) {
                continue;
            }

            $normalized = $this->normalize_iran_phone( $raw_number, $opts['add_country'] );
            if ( $normalized ) {
                $admin_numbers[ $normalized ] = true;
            } else {
                error_log( '[WC WhatsApp] Admin phone normalization failed: ' . $raw_number );
            }
        }

        if ( empty( $admin_numbers ) ) {
            return;
        }

        // آماده‌سازی placeholder های پیام مدیر
        $admin_placeholders = array(
            '{order_id}'        => $order->get_id(),
            '{customer_name}'   => $order->get_billing_first_name() . ' ' . $order->get_billing_last_name(),
            '{customer_phone}'  => $order->get_billing_phone(),
            '{customer_email}'  => $order->get_billing_email(),
            '{shipping_address}' => $this->get_shipping_address( $order ),
            '{products_list}'   => $this->get_products_list( $order ),
            '{total}'           => $this->get_clean_order_total( $order ),
            '{order_date}'      => $order->get_date_created()->format( 'Y-m-d H:i:s' ),
            '{status_text}'     => $this->get_status_text( $current_status ),
            '{status_message}'  => $this->get_status_message( $current_status ),
        );

        $admin_message = strtr( $opts['admin_message'], $admin_placeholders );
        $admin_message = wp_strip_all_tags( $admin_message );
        $admin_message = html_entity_decode( $admin_message, ENT_QUOTES, 'UTF-8' );

        foreach ( array_keys( $admin_numbers ) as $admin_phone ) {
            error_log( '[WC WhatsApp] Sending admin notification to: ' . $admin_phone );

            $admin_result = $this->send_message( $admin_phone, $admin_message, $opts );

            if ( is_wp_error( $admin_result ) ) {
                $this->log_order_note( $order, 'WhatsApp: خطا در ارسال اعلان مدیر برای ' . $admin_phone . ' - ' . $admin_result->get_error_message() );
            } else {
                $this->log_order_note( $order, 'WhatsApp: اعلان مدیر برای ' . $admin_phone . ' ارسال شد. پاسخ API: ' . $admin_result );
            }
        }
    }

    /**
     * Send SMS notification using Melipayamak service
     *
     * @param WC_Order $order
     * @param string   $current_status
     * @param array    $opts
     * @param array    $placeholders
     * @param string   $normalized_phone
     * @return bool
     */
    private function send_sms_notification( $order, $current_status, $opts, $placeholders, $normalized_phone ) {
        if ( $opts['sms_enabled'] !== 'yes' ) {
            return false;
        }

        $sms_statuses = isset( $opts['sms_statuses'] ) ? (array) $opts['sms_statuses'] : array();
        if ( empty( $sms_statuses ) || ! in_array( $current_status, $sms_statuses, true ) ) {
            return false;
        }

        if ( empty( $opts['sms_username'] ) || empty( $opts['sms_password'] ) || empty( $opts['sms_from'] ) ) {
            error_log( '[WC WhatsApp] SMS not sent. Missing credentials or sender number.' );
            return false;
        }

        $sms_number = $this->format_phone_for_sms( $normalized_phone );
        if ( ! $sms_number ) {
            $this->log_order_note( $order, 'SMS: شماره تلفن برای ارسال پیامک معتبر نیست. مقدار: ' . $normalized_phone );
            return false;
        }

        $sms_message = strtr( $opts['sms_message'], $placeholders );
        $sms_message = wp_strip_all_tags( $sms_message );
        $sms_message = html_entity_decode( $sms_message, ENT_QUOTES, 'UTF-8' );

        $params = array(
            'username' => $opts['sms_username'],
            'password' => $opts['sms_password'],
            'to'       => $sms_number,
            'from'     => $opts['sms_from'],
            'text'     => $sms_message,
            'isflash'  => false,
        );

        $response = $this->send_sms_via_melipayamak( $params, $opts['sms_wsdl'] );

        if ( is_wp_error( $response ) ) {
            $this->log_order_note( $order, 'SMS: خطا در ارسال پیامک - ' . $response->get_error_message() );
            return false;
        } else {
            $this->log_order_note( $order, 'SMS: پیامک ارسال شد به ' . $sms_number . '. پاسخ سرویس: ' . $response );
            return true;
        }
    }

    /**
     * Convert normalized phone (98...) to local format (09...)
     *
     * @param string $normalized_phone
     * @return string
     */
    private function format_phone_for_sms( $normalized_phone ) {
        if ( empty( $normalized_phone ) ) {
            return '';
        }

        $digits = preg_replace( '/\D+/', '', $normalized_phone );
        if ( empty( $digits ) ) {
            return '';
        }

        if ( strpos( $digits, '98' ) === 0 ) {
            $digits = '0' . substr( $digits, 2 );
        } elseif ( strpos( $digits, '9' ) === 0 && strlen( $digits ) === 10 ) {
            $digits = '0' . $digits;
        } elseif ( strpos( $digits, '0' ) !== 0 ) {
            $digits = '0' . ltrim( $digits, '0' );
        }

        if ( strlen( $digits ) === 11 && strpos( $digits, '09' ) === 0 ) {
            return $digits;
        }

        return '';
    }

    /**
     * Call Melipayamak SOAP API
     *
     * @param array  $params
     * @param string $wsdl
     * @return string|WP_Error
     */
    private function send_sms_via_melipayamak( $params, $wsdl ) {
        if ( ! class_exists( 'SoapClient' ) ) {
            return new WP_Error( 'sms_missing_extension', 'SoapClient در دسترس نیست. لطفاً افزونه SOAP را فعال کنید.' );
        }

        if ( empty( $wsdl ) ) {
            $wsdl = $this->defaults['sms_wsdl'];
        }

        try {
            ini_set( 'soap.wsdl_cache_enabled', '0' );
            $client = new SoapClient( $wsdl, array( 'encoding' => 'UTF-8', 'exceptions' => true ) );
            $result = $client->SendSimpleSMS2( $params );

            if ( is_object( $result ) && isset( $result->SendSimpleSMS2Result ) ) {
                return $result->SendSimpleSMS2Result;
            }

            return json_encode( $result );
        } catch ( Exception $e ) {
            return new WP_Error( 'sms_exception', $e->getMessage() );
        }
    }

    /**
     * Get customer phone number from order or user data
     *
     * @param WC_Order $order
     * @param array $opts
     * @return string
     */
    private function get_customer_phone( $order, $opts ) {
        $phone_raw = '';

        // اگر auto_search انتخاب شده، در همه جا جستجو کن
        if ( $opts['phone_field'] === 'auto_search' ) {
            // ابتدا از اطلاعات سفارش جستجو کن
            $order_fields = array( 'billing_phone', 'shipping_phone', 'phone', 'mobile' );
            foreach ( $order_fields as $field ) {
                if ( $field === 'billing_phone' && method_exists( $order, 'get_billing_phone' ) ) {
                    $phone_raw = $order->get_billing_phone();
                } else {
                    $phone_raw = $order->get_meta( $field );
                }
                if ( ! empty( $phone_raw ) ) {
                    error_log( '[WC WhatsApp] Phone found in order field: ' . $field . ' = ' . $phone_raw );
                    break;
                }
            }

            // اگر در سفارش پیدا نشد، از اطلاعات کاربر جستجو کن
            if ( empty( $phone_raw ) ) {
                $customer_id = $order->get_customer_id();
                if ( $customer_id > 0 ) {
                    $user_fields = array( 'billing_phone', 'phone', 'mobile', 'user_phone', 'telephone' );
                    foreach ( $user_fields as $field ) {
                        $phone_raw = get_user_meta( $customer_id, $field, true );
                        if ( ! empty( $phone_raw ) ) {
                            error_log( '[WC WhatsApp] Phone found in user meta: ' . $field . ' = ' . $phone_raw );
                            break;
                        }
                    }
                }
            }
        } else {
            // جستجوی معمولی در فیلد مشخص شده
            if ( $opts['phone_field'] === 'billing_phone' && method_exists( $order, 'get_billing_phone' ) ) {
                $phone_raw = $order->get_billing_phone();
            } else {
                $phone_raw = $order->get_meta( $opts['phone_field'] );
            }

            // اگر شماره پیدا نشد، از اطلاعات کاربر جستجو کن
            if ( empty( $phone_raw ) ) {
                $customer_id = $order->get_customer_id();
                if ( $customer_id > 0 ) {
                    $user_phone = get_user_meta( $customer_id, $opts['phone_field'], true );
                    if ( ! empty( $user_phone ) ) {
                        $phone_raw = $user_phone;
                        error_log( '[WC WhatsApp] Phone found in user meta: ' . $phone_raw );
                    }
                }
            }
        }

        return $phone_raw;
    }

    /**
     * Normalize Iranian phone numbers to format: 989XXXXXXXXX
     *
     * Accepts inputs like:
     *  - +989123456789
     *  - 00989123456789
     *  - 09123456789
     *  - 9123456789
     *  - 989123456789
     *
     * Returns normalized digits string or empty string on failure.
     *
     * @param string $input
     * @param string $default_cc default country code (e.g. '98')
     * @return string
     */
    private function normalize_iran_phone( $input, $default_cc = '98' ) {
        if ( empty( $input ) ) return '';

        // نگه‌داشتن فقط ارقام
        $digits = preg_replace( '/\D+/', '', $input );

        if ( empty( $digits ) ) return '';

        // اگر با 0098 شروع شده (مثال: 0098912...)
        if ( strpos( $digits, '0098' ) === 0 ) {
            $digits = substr( $digits, 2 ); // 00 -> remove one leading 00 to get 98...
        }

        // اگر با 0 شروع کنه (مثال: 0912345...)
        if ( strpos( $digits, '0' ) === 0 ) {
            // حذف صفر ابتدایی
            $digits = ltrim( $digits, '0' );
        }

        // اگر با 98 شروع شد — خوب است
        if ( strpos( $digits, $default_cc ) === 0 ) {
            // اگر بعد از 98 یک صفر اضافه بوده و باعث شد طول نادرست شود، از آن جلوگیری نمی‌کنیم چون قبلاً صفر حذف شده
            // مثال درست: 989123456789
            return $digits;
        }

        // اگر با 9 شروع شد (مثال: 9123456789) --> پیشوند 98 اضافه کن
        if ( strpos( $digits, '9' ) === 0 ) {
            return $default_cc . $digits;
        }

        // اگر طول کاملاً کوتاه یا بلند است — تلاش نهایی: اگر طول == 10 و با 9 شروع می‌کند
        if ( strlen( $digits ) === 10 && strpos( $digits, '9' ) === 0 ) {
            return $default_cc . $digits;
        }

        // اگر طول همان 12 و با 98 شروع میکند، آن را قبول کن
        if ( strlen( $digits ) === 12 && strpos( $digits, $default_cc ) === 0 ) {
            return $digits;
        }

        // موارد دیگر را سعی کنیم با الحاق default_cc برطرف کنیم اگر بعد از الحاق طول منطقی شد (12)
        $try = $default_cc . ltrim( $digits, '0' );
        if ( strlen( $try ) === 12 ) {
            return $try;
        }

        // در غیر این صورت شکست
        return '';
    }

    /* ---------- Send message to API ---------- */
    private function send_message( $number, $message, $opts = null ) {
        $opts = $opts ? $opts : $this->get_options();

        $query = array(
            'api_key' => $opts['api_key'],
            'sender'  => $opts['sender'],
            'number'  => $number,
            'message' => $message,
        );

        $url = $opts['api_url'];
        $sep = ( strpos( $url, '?' ) === false ) ? '?' : '&';
        $url .= $sep . http_build_query( $query );

        $args = array(
            'timeout' => 20,
            'headers' => array( 'Accept' => 'application/json' ),
        );

        $response = wp_remote_get( $url, $args );

        if ( is_wp_error( $response ) ) {
            error_log( '[WC WhatsApp] request error: ' . $response->get_error_message() );
            return $response;
        }

        $code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );

        error_log( '[WC WhatsApp] HTTP ' . $code . ' - ' . $body );

        if ( $code >= 200 && $code < 300 ) {
            return $body;
        }

        return new WP_Error( 'wc_whatsapp_error', sprintf( 'API returned status %s', $code ) );
    }

    private function log_order_note( $order, $note ) {
        $opts = $this->get_options();
        if ( $opts['log_to_order'] === 'yes' && $order instanceof WC_Order ) {
            $order->add_order_note( $note, false ); // private note
        }
        error_log( '[WC WhatsApp] ' . $note );
    }
}

new WC_WhatsApp_Sender();

endif;
